//*************************************************************************************************
//
//	Description:
//		Skintest_instanced.fx
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		AMurray				26/10/2007	0.1           	Created
//		TNettleship		23/11/2007	0.2						Changes to support precalculated lighting & shadows.
//	<TABLE>
//
//*************************************************************************************************
#define _SSAO_READY_

#ifndef _3DSMAX_
#define	INSTANCING
#endif

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define USE_SPECULAR
#define NORMAL_MAPPING
#define ENV_MAPPING
#endif

#define MAX_ENVMAP_BIAS	6.0f

// Skinning
#define MAX_BONES 32

#include "genInstancing.fxh"


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif


#if defined(INSTANCING) && defined(_XBOX)
float4 instance_data		// x=no of indices per instance, y=start of index data within IB
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif

#if defined(INSTANCING) && defined(_PS3_)
float4x4 instanceXForm
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float4x4 instanceworldviewproj
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif


//
// Transforms
//

#ifdef INSTANCING
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else

#if defined( _3DSMAX_ )
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

SPECIALISATION_PARAM_DEFAULTS_TRUE( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )
#endif


SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Texture", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 10.0f, 1.0f )
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 1.0f, 1.0f )
#endif



//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

//
// This shader supports variance 3 mask tint
//
#define TINT_USE_TINT_ALWAYS
#include "variance_3mask_tint.fxh"


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


//-----------------------------------------------------------------------
//
// Bones
//

#ifndef _3DSMAX_
#ifndef _PS3_
float4x3 boneTransforms[ MAX_BONES ] : boneTransforms
<
	bool appEdit = false;
>;
#endif
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structures
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float  alpha    : TEXCOORD2;
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal
#ifndef _PS3_	
	float4 boneWeights : BLENDWEIGHT;											// Bone weights
	int4   boneIndices : BLENDINDICES;										// Bone indices
#endif	

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif

#if !defined(_PS3_)	// Not needed on PS3
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position

#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float  alpha    : TEXCOORD2;
#else
#	ifdef USE_VERTEXALPHAHIDING
		float4 colour   : COLOR0;															// Vertex colour
#	endif
#endif

#ifndef _PS3_
	float4 boneWeights : BLENDWEIGHT;											// Bone weights
	int4   boneIndices : BLENDINDICES;										// Bone indices
#endif	

#if !defined(_PS3_)	// Not needed on PS3
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};


// Output structures
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD5;														// Vertex colour
	float4 tintR			: TEXCOORD6;
	float4 texCoordAndAtlasUV			: TEXCOORD0;
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	float4 tintG			: TEXCOORD3;
	float4 tintB			: TEXCOORD4;
	
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD7;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD8;												// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 9 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 7 )
#endif
};


struct SHADOWGEN_VSOUTPUT
{
	float4 position		 : POSITION;												// View-coords position
#ifdef NEEDS_SHADOW_COORDS
	float4 shadowCoord : TEXCOORD0;												// UV coords for texture channel 0
#endif
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position		 : POSITION;												// View-coords position
	float4 coords			: TEXCOORD0;
};


#include "atlasing.fxh"

#ifdef _3DSMAX_
float subTextureIndex 
<
	string UIName = "[MaxOnly] Sub Texture Index";
	string UIWidget = "Spinner";
	float UIStep = 1.0;
	bool appEdit = false;
	bool export = false;
> = 0.0;
#endif

float GetAtlasSubTextureIndex(int generalizedInstanceConstantIndex )
{
#ifdef _3DSMAX_
	return subTextureIndex;
#else
	return GenI_GetFloat4( generalizedInstanceConstantIndex + 3).r;
#endif
}


#ifdef _3DSMAX_
float alphaThreshold0
<
	string UIName = "[MaxOnly] Render verts where alpha <=";
	string UIWidget = "Spinner";
	float UIMin = 0.0;
	float UIMax = 1.0;
	bool appEdit = false;
	bool export = false;
> = 0.0;

float alphaThreshold1
<
	string UIName = "[MaxOnly] Also verts where alpha >=";
	string UIWidget = "Spinner";
	float UIMin = 0.0;
	float UIMax = 1.0;
	bool appEdit = false;
	bool export = false;
> = 1.0;
#endif


float GetAlphaThreshold0(int generalizedInstanceConstantIndex )
{
#ifdef _3DSMAX_
	return alphaThreshold0;
#else
	return GenI_GetFloat4( generalizedInstanceConstantIndex + 3).b;
#endif
}

float GetAlphaThreshold1(int generalizedInstanceConstantIndex )
{
#ifdef _3DSMAX_
	return alphaThreshold1;
#else
	return GenI_GetFloat4( generalizedInstanceConstantIndex + 3).a;
#endif
}

#include "vertexAlphaHiding.fxh"


//-----------------------------------------------------------------------
//
// Vertex shader code
//
// TODO: rearrange to transform eye and lights into local space before
// vector calculations.


#ifdef _POST_PARSE_
VSOUTPUT SkintestVertexShader( VSINPUT _input )
{

#else

#if defined(INSTANCING) && defined(_XBOX)
VSOUTPUT SkintestVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fnormal;
	float4 fboneweights;
	int4   fboneindices;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	float4 ftangent;
	float4 fbinormal;
#endif
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fnormal, iMeshIndex, normal0;
        vfetch fboneweights, iMeshIndex, blendweight0;
        vfetch fboneindices, iMeshIndex, blendindices0;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
        vfetch ftangent, iMeshIndex, tangent0;
        vfetch fbinormal, iMeshIndex, binormal0;
#endif

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.normal = fnormal.xyz;
	_input.boneWeights = fboneweights;
	_input.boneIndices = fboneindices;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	_input.tangent = fboneweights;
	_input.binormal = fbinormal.xyz;
#endif
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;


#else

VSOUTPUT SkintestVertexShader( VSINPUT _input )
{

#endif
#endif

	VSOUTPUT _output;

#ifdef INSTANCING
	#ifdef _PS3_
		float4x4	worldlocal = instanceXForm;
	#else
		float4x4	worldlocal = { _input.world1, _input.world2, _input.world3, _input.world4 };
	#endif	
#else
	float4x4	worldlocal = world;
#endif

	// extract the constant index to be used to fetch this instance variation data
	int generalizedInstanceConstantIndex = worldlocal[3][3];
	// and unhack the world matrix back to proper world matrix
	worldlocal[3][3] = 1.0f;

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( worldlocal, viewproj );
#endif
#endif

	// Perform skinning:
	//	Using ***object space*** bone xforms, xform the position, normal,
	//	tangent and binormal vectors using the weights and indices for
	//	this vertex.
	float3 skinnedPos = 0.0f;
	float3 skinnedNormal = 0.0f;
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		float3 skinnedTangent = 0.0f;
		float3 skinnedBinormal = 0.0f;
#endif
	DEPENDENT_CODE_END( useNormalMap )

#ifndef _3DSMAX_

#ifndef _PS3_
	// cast the vectors to arrays for use in the for loop below
	float blendWeightsArray[ 4 ] = ( float[ 4 ] )_input.boneWeights;
	float totalWeight = 0.0f;

	// calculate the pos/tangent space using the "normal" weights
	for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
	{
		float weight = blendWeightsArray[ boneIndex ];
		if ( weight > 0.0f )
		{
			totalWeight += weight;
 
			skinnedPos += mul( float4( _input.position, 1.0f ), boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
			skinnedNormal += mul( _input.normal, boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
			DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
				skinnedTangent += mul( _input.tangent, boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
				skinnedBinormal += mul( _input.binormal, boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
#endif
			DEPENDENT_CODE_END( useNormalMap )
		}
	}
	totalWeight = 1.0f / totalWeight;

	// Calculate the weighted average of the influences
	skinnedPos *= totalWeight;
	skinnedNormal *= totalWeight;
#else
	skinnedPos=_input.position;
	skinnedNormal=_input.normal;
#endif	
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
#ifndef _PS3_
		skinnedTangent *= totalWeight;
		skinnedBinormal *= totalWeight;
#else
		skinnedTangent=_input.tangent;
		skinnedBinormal=_input.binormal;
#endif		
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoordAndAtlasUV = float4( _input.texCoord.x, _input.texCoord.y, GetAtlasSubTextureIndex(generalizedInstanceConstantIndex), 0.0 );

	// Calculate clip-space position of the vertex
#ifdef _3DSMAX_
	_input.colour.a = _input.alpha;
#endif
	_output.position = PerformVertexAlphaHiding( mul( float4( skinnedPos, 1.0f ), worldviewproj ), 
#ifdef USE_VERTEXALPHAHIDING
			_input.colour.a, 
#else
			1.0,
#endif	
	GetAlphaThreshold0( generalizedInstanceConstantIndex ), GetAlphaThreshold1( generalizedInstanceConstantIndex ) );

	// Calculate vert's world position
	float3 worldPos = mul( float4( skinnedPos, 1.0f ), worldlocal ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( skinnedNormal, 0.0f ), worldlocal ).xyz );
	_output.normal = normal;

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( skinnedTangent, 0.0f ), worldlocal ).xyz;
		_output.binormal = mul( float4( skinnedBinormal, 0.0f ), worldlocal ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

#else
	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoordAndAtlasUV = float4( _input.texCoord.x, _input.texCoord.y, GetAtlasSubTextureIndex(generalizedInstanceConstantIndex), 0.0 );

	// Calculate clip-space position of the vertex
#ifdef _3DSMAX_
	_input.colour.a = _input.alpha;
#endif
	_output.position = PerformVertexAlphaHiding( mul( float4( _input.position, 1.0f ), worldviewproj ), 
#ifdef USE_VERTEXALPHAHIDING
			_input.colour.a, 
#else
			1.0,
#endif	
	GetAlphaThreshold0( generalizedInstanceConstantIndex ), GetAlphaThreshold1( generalizedInstanceConstantIndex ) );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), worldlocal ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), worldlocal ).xyz );
	_output.normal = normal;
	_output.tangent  = mul( float4( _input.tangent, 0.0f ), worldlocal ).xyz;
	_output.binormal = mul( float4( _input.binormal, 0.0f ), worldlocal ).xyz;
#endif		// #ifndef _3DSMAX_

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 1.0f;
	float maxSpecPowerValue = 1.0f;
#else
	DEPENDENT_CODE_START( useSpecular )
	float globalSpecularFactorValue = globalSpecularFactor;
	DEPENDENT_CODE_END( useSpecular )
#endif

	_output.tintR = GenI_GetFloat4( generalizedInstanceConstantIndex );
	_output.tintG = GenI_GetFloat4( generalizedInstanceConstantIndex + 1);
	_output.tintB = GenI_GetFloat4( generalizedInstanceConstantIndex + 2);

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



#if defined(INSTANCING) && defined(_XBOX)
SHADOWGEN_VSOUTPUT SkintestShadowGenVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
#ifdef USE_VERTEXALPHAHIDING
	float4 fcolour;
#endif
	float4 ftexcoord;
	float4 fboneweights;
	int4   fboneindices;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
#ifdef USE_VERTEXALPHAHIDING
        vfetch fcolour, iMeshIndex, color0;
#endif
        vfetch ftexcoord, iMeshIndex, texcoord0;
        vfetch fboneweights, iMeshIndex, blendweight0;
        vfetch fboneindices, iMeshIndex, blendindices0;

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
#ifdef USE_VERTEXALPHAHIDING
	_input.colour = fcolour;
#endif
	_input.texCoord = ftexcoord.xy;
	_input.boneWeights = fboneweights;
	_input.boneIndices = fboneindices;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
SHADOWGEN_VSOUTPUT SkintestShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
#endif

	SHADOWGEN_VSOUTPUT _output;

#ifdef INSTANCING
	#ifdef _PS3_
		float4x4	worldlocal = instanceXForm;
	#else
		float4x4	worldlocal = { _input.world1, _input.world2, _input.world3, _input.world4 };
	#endif	
#else
	float4x4	worldlocal = world;
#endif

	// extract the constant index to be used to fetch this instance variation data
	int generalizedInstanceConstantIndex = worldlocal[3][3];
	// and unhack the world matrix back to proper world matrix
	worldlocal[3][3] = 1.0f;

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( worldlocal, viewproj );
#endif
#endif

	// Perform skinning:
	//	Using ***object space*** bone xforms, xform the position, normal,
	//	tangent and binormal vectors using the weights and indices for
	//	this vertex.
	float3 skinnedPos = 0.0f;

#ifndef _3DSMAX_

#ifndef _PS3_
	// cast the vectors to arrays for use in the for loop below
	float blendWeightsArray[ 4 ] = ( float[ 4 ] )_input.boneWeights;
	float totalWeight = 0.0f;

	// calculate the pos/tangent space using the "normal" weights
	for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
	{
		float weight = blendWeightsArray[ boneIndex ];
		if ( weight > 0.0f )
		{
			totalWeight += weight;

			skinnedPos += mul( float4( _input.position, 1.0f ), boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
		}
	}
	totalWeight = 1.0f / totalWeight;

	// Calculate the weighted average of the influences
	skinnedPos *= totalWeight;
#else
	skinnedPos=_input.position;
#endif	
#endif

#ifdef _3DSMAX_
	_input.colour.a = _input.alpha;
#endif
	_output.position = PerformVertexAlphaHiding( mul( float4( skinnedPos, 1.0f ), worldviewproj ), 
#ifdef USE_VERTEXALPHAHIDING
			_input.colour.a, 
#else
			1.0,
#endif
	GetAlphaThreshold0( generalizedInstanceConstantIndex ), GetAlphaThreshold1( generalizedInstanceConstantIndex ) );

#ifdef NEEDS_SHADOW_COORDS
	OUTPUT_SHADOW_COORDS;
#endif

	return _output;
}


#if defined(INSTANCING) && defined(_XBOX)
ZPRIMEDOF_VSOUTPUT SkintestZPrimeDOFVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
#ifdef USE_VERTEXALPHAHIDING
	float4 fcolour;
#endif
	float4 ftexcoord;
	float4 fboneweights;
	int4   fboneindices;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
#ifdef USE_VERTEXALPHAHIDING
        vfetch fcolour, iMeshIndex, color0;
#endif
        vfetch ftexcoord, iMeshIndex, texcoord0;
        vfetch fboneweights, iMeshIndex, blendweight0;
        vfetch fboneindices, iMeshIndex, blendindices0;

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
#ifdef USE_VERTEXALPHAHIDING
	_input.colour = fcolour;
#endif
	_input.texCoord = ftexcoord.xy;
	_input.boneWeights = fboneweights;
	_input.boneIndices = fboneindices;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
ZPRIMEDOF_VSOUTPUT SkintestZPrimeDOFVertexShader
	( 
			SHADOWGEN_VSINPUT _input 
	)
{
#endif
	ZPRIMEDOF_VSOUTPUT _output;

#ifdef INSTANCING
	#ifdef _PS3_
		float4x4	worldlocal = instanceXForm;
	#else
		float4x4	worldlocal = { _input.world1, _input.world2, _input.world3, _input.world4 };
	#endif	
#else
	float4x4	worldlocal = world;
#endif

	// extract the constant index to be used to fetch this instance variation data
	int generalizedInstanceConstantIndex = worldlocal[3][3];
	// and unhack the world matrix back to proper world matrix
	worldlocal[3][3] = 1.0f;

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( worldlocal, viewproj );
#endif	
#endif

	// Perform skinning:
	//	Using ***object space*** bone xforms, xform the position, normal,
	//	tangent and binormal vectors using the weights and indices for
	//	this vertex.
	float3 skinnedPos = 0.0f;

#ifndef _3DSMAX_

#ifndef _PS3_
	// cast the vectors to arrays for use in the for loop below
	float blendWeightsArray[ 4 ] = ( float[ 4 ] )_input.boneWeights;
	float totalWeight = 0.0f;

	// calculate the pos/tangent space using the "normal" weights
	for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
	{
		float weight = blendWeightsArray[ boneIndex ];
		if ( weight > 0.0f )
		{
			totalWeight += weight;

			skinnedPos += mul( float4( _input.position, 1.0f ), boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
		}
	}
	totalWeight = 1.0f / totalWeight;

	// Calculate the weighted average of the influences
	skinnedPos *= totalWeight;
#else
	skinnedPos=_input.position;
#endif	
#endif

#ifdef _3DSMAX_
	_input.colour.a = _input.alpha;
#endif
	_output.position = PerformVertexAlphaHiding( mul( float4( skinnedPos, 1.0f ), worldviewproj ), 
#ifdef USE_VERTEXALPHAHIDING
			_input.colour.a, 
#else
			1.0,
#endif
	 GetAlphaThreshold0( generalizedInstanceConstantIndex ), GetAlphaThreshold1( generalizedInstanceConstantIndex ) );
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD5;										// Vertex colour
	float4 tintR			: TEXCOORD6;
	float4 texCoordAndAtlasUV		: TEXCOORD0;						// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;								// Normal vector (world space)
	float4 eye				: TEXCOORD2;								// Eye vector (world space)

	float4 tintG			: TEXCOORD3;
	float4 tintB			: TEXCOORD4;

	float3 tangent		: TEXCOORD7;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD8;								// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 9 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD5;														// Vertex colour
	float4 tintR			: TEXCOORD6;
	float4 texCoordAndAtlasUV		: TEXCOORD0;						// UV coords for texture channel 0
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)
	float4 tintG			: TEXCOORD3;
	float4 tintB			: TEXCOORD4;
	
#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD7_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD8_centroid;								// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 9 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 7 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


struct SHADOWGEN_PSINPUT
{
#if defined(NEEDS_SHADOW_COORDS) || defined(_PS3_)
	float4 shadowCoord : TEXCOORD0;
#endif
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

float4 TintIfEnabled( PSINPUT input, float4 hiresColour, float2 maskTexCoord, uniform bool lowDetail )
{
#if defined( _3DSMAX_ )
	return Tint_DoIfEnabledEx( hiresColour, maskTexCoord, tint_tintR, tint_tintG, tint_tintB, lowDetail );
#else
	return Tint_DoIfEnabledEx( hiresColour, maskTexCoord, input.tintR.rgb, input.tintG.rgb, input.tintB.rgb, lowDetail );
#endif
}

//REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float2 atlasTexCoord = texCoord2DAtlas( _input.texCoordAndAtlasUV.xy, _input.texCoordAndAtlasUV.z );
	float4 diffuseTexColour = TintIfEnabled( _input, tex2D( diffuseMap, atlasTexCoord ), atlasTexCoord, false );
	float4 specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );

	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
#else
	DEPENDENT_CODE_START( useSpecular )
	globalSpecularFactorValue = globalSpecularFactor;
	minSpecPowerValue = minSpecPower;
	maxSpecPowerValue = maxSpecPower;
	DEPENDENT_CODE_END( useSpecular )
#endif

	DEPENDENT_CODE_START( useSpecular )
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
		// Read specular texture
		specularTexColour = tex2D( specularMap, atlasTexCoord );

		DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
			globalSpecularFactorValue = 0.0f;
#endif
		DEPENDENT_CODE_END( useEnvironmentMap )

#endif
	DEPENDENT_CODE_ELSE( useSpecular )
#if defined( _3DSMAX_ ) || !defined( USE_SPECULAR )
		// No specular, so default the colour to ones (will be optimised out)
		specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );

		globalSpecularFactorValue = 0.0f;
		minSpecPowerValue = 0.0f;
		maxSpecPowerValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )
	
	// Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, atlasTexCoord );
		float3 normalFromMap;

		normalFromMap.rgb = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the reflection vector
		float3 worldReflectionVector = reflect( -eye, normal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( worldReflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( worldReflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor;

		// Switch specular lighting off ... the envmap provides our specular
		globalSpecularFactorValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT output;
	
#if !defined(NEEDS_SHADOW_COORDS)
	output.Colour=0;
#else
	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
#endif	

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	output.Colour = _input.coords.z / _input.coords.w;

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestOverdrawFragmentShader( PSINPUT _input )
{
	PSOUTPUT output;

	output.Colour = 0.05f;

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestLowDetailFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float2 atlasTexCoord = texCoord2DAtlas( _input.texCoordAndAtlasUV.xy, _input.texCoordAndAtlasUV.z );
	float4 diffuseTexColour = TintIfEnabled( _input, tex2D( diffuseMap, atlasTexCoord ), atlasTexCoord, true );
	float4 specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal = TSnormal;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the reflection vector
		float3 worldReflectionVector = reflect( -eye, normal );

		// Fetch the environment map colour using the world coords vector
		float4 environmentTexColour = texCUBE( environmentMap, worldReflectionVector );

		// Calculate envmap colour and add to diffuse
		accumulatedColour += environmentTexColour * globalEMapFactor;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	accumulatedColour.w = diffuseTexColour.w * _input.colour.w;
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique skintest
<
	int maxbones = MAX_BONES;
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool instancedShader = true;
	bool skinOnSpu=true;

	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_skintest_zprime_dof";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_skintest_shadowgen";
	int    shadowGenDeferredID	= 0;

	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "_skintest_lowdetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestFragmentShader();
#endif		
	}
}


technique skintest_translucent
<
	int maxbones = MAX_BONES;
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool instancedShader = true;
	bool skinOnSpu=true;

	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest_translucent";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_skintest_shadowgen";
	int    shadowGenDeferredID	= 0;
	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "_skintest_lowdetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestFragmentShader();
#endif		
	}
}


technique skintest_alphatest
<
	int maxbones = MAX_BONES;
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool instancedShader = true;
	bool skinOnSpu=true;

	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest_alphatest";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_DONT_RENDER";

	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "_skintest_lowdetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 128;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 128;
		AlphaFunc = GreaterEqual;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestFragmentShader();
#endif		
	}
}


technique _skintest_shadowgen
<
	int maxbones = MAX_BONES;
	bool preservesGlobalState = true;
	bool skinOnSpu=true;
>
{
	pass Pass0
	{
		AlphaTestEnable = false;
#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestShadowGenVertexShader();
		PixelShader = compile sce_fp_rsx SkintestShadowGenFragmentShader();
#else		
	#if defined(_XBOX) && !defined(SHADOWMAP_COLOUR)
		VertexShader = compile vs_3_0 SkintestShadowGenVertexShader();
		PixelShader = null;
	#else
		VertexShader = compile vs_3_0 SkintestShadowGenVertexShader();
		PixelShader = compile ps_3_0 SkintestShadowGenFragmentShader();
	#endif
#endif		
	}
}

technique _skintest_zprime_dof
<
	int maxbones = MAX_BONES;
	bool preservesGlobalState = true;
	bool skinOnSpu=true;
>
{
	pass Pass0
	{
#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestZPrimeDOFVertexShader();
		PixelShader = compile sce_fp_rsx SkintestZPrimeDOFFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestZPrimeDOFVertexShader();
		PixelShader = compile ps_3_0 SkintestZPrimeDOFFragmentShader();
#endif		
	}
}

technique _skintest_overdraw
<
	int maxbones = MAX_BONES;
	bool skinOnSpu=true;
>
{
	pass Pass0
	{
		ZEnable = false;
		ZWriteEnable = false;
		AlphaBlendEnable = true;

#if defined (_PS3_)
		BlendEquation=int(FuncAdd);
		BlendFunc=int2(One, One);
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestOverdrawFragmentShader();
#else		
		SrcBlend = ONE;
		DestBlend = ONE;
		BlendOp = ADD;
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestOverdrawFragmentShader();
#endif		
	}
}


technique _skintest_lowdetail
<
	int maxbones = MAX_BONES;
	bool preservesGlobalState = true;
	bool skinOnSpu=true;
>
{
	pass Pass0
	{
#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestLowDetailFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestLowDetailFragmentShader();
#endif		
	}
}
